//*************************************************************************************************
//
//	Description:
//		textured_skydome.fx - Shader for textured HDR skydomes.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     26/11/2007  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"
#include "lighting_globals.fxh"


// Fog/sky multiplier needs to be higher on 360 to get the same end result
// Update - with other changes it can now be the same on all platforms, but left them separate for now just in case
#if defined(_XBOX)
#define FOGGING_FALLOFF_MULTIPLIER 3.0
#else
#define FOGGING_FALLOFF_MULTIPLIER 3.0
#endif
#define FOGGING_FALLOFF_POWER 2.0


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define EDIT_ME_TO_BOOST_ENVMAP_BRIGHTNESS 2.2


// Envoding type
//#define _ENCODING_LUVW_
#define _ENCODING_RGBDIV8_
//#define _ENCODING_RGBE_



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif

#ifdef _3DSMAX_
float skyBrightness
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float envBrightness
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
float skyBrightness
<
	bool appEdit = true;
>;

float envBrightness
<
	bool appEdit = true;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
>;

float4x4 world : World
<
	bool appEdit = false;
	bool dynamic = true;
>;
#endif



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture colourTexture : DiffuseMap						// Colour in RGB
#else
texture colourTexture : TEXTURE							// Colour in RGB
#endif
<
	string UIName = "Colour Tex {UV1}";
	bool appEdit = true;
>;

texture intensityTexture : TEXTURE						// Intensity in R
<
	string UIName = "Intensity Tex {UV1}";
	bool appEdit = true;
>;



//
// Lighting
//

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D colourMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="colourTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < colourTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


sampler2D intensityMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="intensityTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < intensityTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};



//
// Functions
//



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float  fogScale		: TEXCOORD1;												// Fog colour scale
};

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float  fogScale		: TEXCOORD1;												// Fog colour scale
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT TexturedSkydomeVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4 worldviewproj = mul( world, viewproj );
#endif

	// Calculate clip-space position of the vertex (forced to the farplane)
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj ).xyww;

	// If we don't subtract a small epsilon from the z, some graphics cards' inaccurate
	// calcs result in pixels bing clipped past the farZ
	_output.position.z -= 0.0001f;

	// Calculate the output fog scale (scaled by the square of the distance from the centre of the texture)
	float2 centredTexCoord = ( _input.texCoord * 2.0f ) - 1.0f;
	_output.fogScale = dot( centredTexCoord, centredTexCoord );
//	_output.fogScale *= _output.fogScale;

	_output.fogScale = pow( _output.fogScale, FOGGING_FALLOFF_POWER ) * FOGGING_FALLOFF_MULTIPLIER;

	// Copy simple invariant params to output structure
	_output.texCoord = _input.texCoord;

	return _output;
}


VSOUTPUT_LD TexturedSkydomeLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4 worldviewproj = mul( world, viewproj );
#endif

	// Calculate clip-space position of the vertex (forced to the farplane)
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj ).xyww;

	// If we don't subtract a small epsilon from the z, some graphics cards' inaccurate
	// calcs result in pixels bing clipped past the farZ
	_output.position.z -= 0.00001f;

	// Copy simple invariant params to output structure
	_output.texCoord = _input.texCoord;

	// Calculate the output fog scale (scaled by the square of the distance from the centre of the texture)
	float2 centredTexCoord = ( _input.texCoord * 2.0f ) - 1.0f;
	_output.fogScale = dot( centredTexCoord, centredTexCoord );
	_output.fogScale = pow( _output.fogScale, FOGGING_FALLOFF_POWER ) * FOGGING_FALLOFF_MULTIPLIER;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

struct PSINPUT
{
	float2 texCoord : TEXCOORD0;						// UV coords for texture channel 0
	float  fogScale	: TEXCOORD1;						// Fog colour scale
};

struct PSINPUT_LD
{
	float2 texCoord : TEXCOORD0;						// UV coords for texture channel 0
	float  fogScale	: TEXCOORD1;						// Fog colour scale
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT TexturedSkydomeFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 texColour = tex2D( colourMap, _input.texCoord );

#if defined( _ENCODING_LUVW_ )
	float4 texIntensity = tex2D( intensityMap, _input.texCoord );
	texColour *= texIntensity.r;
	texColour.a = 1.0f;
#elif defined( _ENCODING_RGBDIV8_ )
	texColour.rgb *= 1.0f / texColour.a;
	texColour.a = 1.0f;
#elif defined( _ENCODING_RGBE_ )
	texColour.rgb = pow( texColour.rgb, texColour.a );
	texColour.a = 1.0f;
#endif

	APPLY_DISTANCE_FOG( texColour, _input.fogScale );
	texColour.rgb*=skyBrightness;
	_output.Colour = CalculateOutputPixel( texColour );

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT TexturedSkydomeLowDetailFragmentShader( PSINPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 texColour = tex2D( colourMap, _input.texCoord );

#if defined( _ENCODING_LUVW_ )
	float4 texIntensity = tex2D( intensityMap, _input.texCoord );
	texColour *= texIntensity.r;
	texColour.a = 1.0f;
#elif defined( _ENCODING_RGBDIV8_ )
	texColour.rgb *= 1.0f / texColour.a;
	texColour.a = 1.0f;
#elif defined( _ENCODING_RGBE_ )
	texColour.rgb = pow( texColour.rgb, texColour.a );
	texColour.a = 1.0f;
#endif

	APPLY_DISTANCE_FOG( texColour, _input.fogScale );

	texColour.rgb *= envBrightness;

	_output.Colour = CalculateLowDetailOutputPixel( texColour );

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique textured_skydome
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "textured_skydome";
	int    normalDeferredID		= 1;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Textured_Skydome_LowDetail";
	int    lowDetailDeferredID = 1;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx TexturedSkydomeVertexShader();
		PixelShader = compile sce_fp_rsx TexturedSkydomeFragmentShader();
#else
		VertexShader = compile vs_3_0 TexturedSkydomeVertexShader();
		PixelShader = compile ps_3_0 TexturedSkydomeFragmentShader();
#endif
	}
}


// Inherits renderstate from the high detail technique
technique _Textured_Skydome_LowDetail
<
  bool preservesGlobalState = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx TexturedSkydomeLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx TexturedSkydomeLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 TexturedSkydomeLowDetailVertexShader();
		PixelShader = compile ps_3_0 TexturedSkydomeLowDetailFragmentShader();
#endif		
	}
}
